/*
 * Copyright 2008 Sony Corporation
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 *
 *   * Redistributions of source code must retain the above copyright notice,
 *     this list of conditions and the following disclaimer.
 *   * Redistributions in binary form must reproduce the above copyright
 *     notice, this list of conditions and the following disclaimer in the
 *     documentation and/or other materials provided with the distribution.
 *   * Neither the names of the copyright holders nor the names of their
 *     contributors may be used to endorse or promote products derived from this
 *     software without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
 * AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS BE
 * LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR
 * CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF
 * SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
 * INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN
 * CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE)
 * ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "e_cell.h"
#include "e_cell_err.h"

#include <string.h>

#define GET_CELL_MD_CTX(evp_md_ctx) \
	((cell_md_ctx_t *)((evp_md_ctx)->md_data))

static void spe_digest_request_init(spe_digest_request_t *cmd, EVP_MD_CTX *ctx)
	{
	cell_md_ctx_t *cell_md = GET_CELL_MD_CTX(ctx);

	cmd->request = 0;
	cmd->nid = ctx->digest->type;
	cmd->size = 0;
	cell_md->in_buffer_ptr %= (cell_md->queue_depth + 2);
	cmd->in_ea =
		(uintptr_t)cell_md->in_buffer +
		cell_md->in_buffer_ptr * cell_md->unit_size;
	cell_md->in_buffer_ptr++;
	}

static int cell_digest_copy(EVP_MD_CTX *to,const EVP_MD_CTX *from)
	{
	CELLerr(CELL_F_CELL_DIGEST_COPY,CELL_R_NOT_IMPLEMENTED);

	return 0;
	}

static int cell_digest_cleanup(EVP_MD_CTX *ctx)
	{
	cell_md_ctx_t *cell_md = GET_CELL_MD_CTX(ctx);

	DPRINTF("\n");

	if (cell_md->spe)
		{
		/* send 'QUIT' command */
		spe_digest_request_t cmd;
		spe_digest_request_init(&cmd, ctx);
		cmd.request = SPE_DIGEST_COMMAND_QUIT;
		cell_queue_push(cell_md->q_in, &cmd);

		/* wait for exiting */
		spe_task_wait(cell_md->spe);
		}

	if (cell_md->q_in)
		{
		cell_queue_destroy(cell_md->q_in);
		cell_md->q_in = 0;
		}
	if (cell_md->q_out)
		{
		cell_queue_destroy(cell_md->q_out);
		cell_md->q_out = 0;
		}
	if (cell_md->in_buffer)
		{
		OPENSSL_cleanse(cell_md->in_buffer,
			cell_md->unit_size * (cell_md->queue_depth + 2));
		aligned_free(cell_md->in_buffer);
		cell_md->in_buffer = NULL;
		}
	if (cell_md->spe)
		{
		spe_task_destroy(cell_md->spe);
		cell_md->spe = NULL;
		}

	return 1;
	}

static int cell_digest_init(EVP_MD_CTX *ctx, spe_program_handle_t *prog)
	{
	cell_md_ctx_t *cell_md = GET_CELL_MD_CTX(ctx);
	spe_digest_request_t cmd;

	DPRINTF("\n");

	/* clear */
	memset(cell_md, 0, sizeof(*cell_md));

	/* copy global parameters */
	CRYPTO_r_lock(CRYPTO_LOCK_ENGINE);
	cell_md->queue_depth = cell_queue_depth;
	cell_md->unit_size = cell_unit_size;
	CRYPTO_r_unlock(CRYPTO_LOCK_ENGINE);

	/* create task object */
	cell_md->spe = spe_task_create(prog);
	if (!cell_md->spe)
		{
		cell_digest_cleanup(ctx);
		return 0;
		}

	/* allocate buffers */
	cell_md->in_buffer =
		aligned_malloc(SHARED_DATA_ALIGN,
			cell_md->unit_size * (cell_md->queue_depth + 2));
	if (!cell_md->in_buffer)
		{
		cell_digest_cleanup(ctx);
		CELLerr(CELL_F_CELL_DIGEST_INIT, CELL_R_MEMORY_ALLOCATION_FAILED);
		return 0;
		}

	/* create queues */
	cell_md->q_in =
		cell_queue_create(cell_md->spe,
			sizeof(spe_digest_request_t), cell_md->queue_depth, 0);
	if (!cell_md->q_in)
		{
		cell_digest_cleanup(ctx);
		return 0;
		}

	cell_md->q_out =
		cell_queue_create(cell_md->spe,
			sizeof(spe_digest_result_t), 1/* depth */, 1);
	if (!cell_md->q_out)
		{
		cell_digest_cleanup(ctx);
		return 0;
		}

	/* run SPE program */
	if (!spe_task_run(cell_md->spe, cell_md->q_in, cell_md->q_out))
		{
		cell_digest_cleanup(ctx);
		return 0;
		}

	/* send 'init' command */
	spe_digest_request_init(&cmd, ctx);
	cmd.request = SPE_DIGEST_COMMAND_INIT;
	cell_queue_push(cell_md->q_in, &cmd);

	return 1;
	}

static int cell_digest_update(EVP_MD_CTX *ctx, const void *data, size_t count)
	{
	cell_md_ctx_t *cell_md = GET_CELL_MD_CTX(ctx);

	DPRINTF("%zu bytes\n", count);

	while (count > 0)
		{
		spe_digest_request_t cmd;
		size_t chunk = count < cell_md->unit_size ? count : cell_md->unit_size;

		DPRINTF("chunk: %zu bytes\n", chunk);

		spe_digest_request_init(&cmd, ctx);
		cmd.request = SPE_DIGEST_COMMAND_UPDATE;
		cmd.size = chunk;
		memcpy((void*)(uintptr_t)cmd.in_ea, data, chunk);
		cell_queue_push(cell_md->q_in, &cmd);

		data += chunk;
		count -= chunk;
		}

	return 1;
	}

static int cell_digest_final(EVP_MD_CTX *ctx, unsigned char *md)
	{
	cell_md_ctx_t *cell_md = GET_CELL_MD_CTX(ctx);
	spe_digest_request_t cmd;
	spe_digest_result_t result;

	DPRINTF("\n");

	/* send 'final' command */
	spe_digest_request_init(&cmd, ctx);
	cmd.request = SPE_DIGEST_COMMAND_FINAL;
	cell_queue_push(cell_md->q_in, &cmd);

	/* read result from output queue */
	cell_queue_pop(cell_md->q_out, &result);
	memcpy(md, result.digest, ctx->digest->md_size);

	cell_digest_cleanup(ctx);

	return 1;
	}


/*** digest algorithm specific ***/

/* MD2 */
#ifndef OPENSSL_NO_MD2
#include <openssl/md2.h>

extern spe_program_handle_t spe_md2_elf;

static int cell_md2_init(EVP_MD_CTX *ctx)
	{
	DPRINTF("\n");

	return cell_digest_init(ctx, &spe_md2_elf);
	}

static const EVP_MD cell_md2 =
	{
	NID_md2,
	NID_md2WithRSAEncryption,
	MD2_DIGEST_LENGTH,
	0,
	cell_md2_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	MD2_BLOCK,
	sizeof(cell_md_ctx_t),
	};
#endif /* !OPENSSL_NO_MD2 */

/* MD4 */
#ifndef OPENSSL_NO_MD4
#include <openssl/md4.h>

extern spe_program_handle_t spe_md4_elf;

static int cell_md4_init(EVP_MD_CTX *ctx)
	{
	DPRINTF("\n");

	return cell_digest_init(ctx, &spe_md4_elf);
	}

static const EVP_MD cell_md4 =
	{
	NID_md4,
	NID_md4WithRSAEncryption,
	MD4_DIGEST_LENGTH,
	0,
	cell_md4_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	MD4_CBLOCK,
	sizeof(cell_md_ctx_t),
	};
#endif /* !OPENSSL_NO_MD4 */

/* MD5 */
#ifndef OPENSSL_NO_MD5
#include <openssl/md5.h>

extern spe_program_handle_t spe_md5_elf;

static int cell_md5_init(EVP_MD_CTX *ctx)
	{
	DPRINTF("\n");

	return cell_digest_init(ctx, &spe_md5_elf);
	}

static const EVP_MD cell_md5 =
	{
	NID_md5,
	NID_md5WithRSAEncryption,
	MD5_DIGEST_LENGTH,
	0,
	cell_md5_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	MD5_CBLOCK,
	sizeof(cell_md_ctx_t),
	};
#endif /* !OPENSSL_NO_MD5 */

/* SHA (SHA0) */
#if !defined(OPENSSL_NO_SHA) && !defined(OPENSSL_NO_SHA0)

#include <openssl/sha.h>

extern spe_program_handle_t spe_sha_elf;

static int cell_sha_init(EVP_MD_CTX *ctx)
	{
	DPRINTF("\n");

	return cell_digest_init(ctx, &spe_sha_elf);
	}

static const EVP_MD cell_sha =
	{
	NID_sha,
	NID_shaWithRSAEncryption,
	SHA_DIGEST_LENGTH,
	0,
	cell_sha_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	SHA_CBLOCK,
	sizeof(cell_md_ctx_t),
	};
#endif /* !OPENSSL_NO_SHA && !OPENSSL_NO_SHA0 */

/* SHA1 */
#ifndef OPENSSL_NO_SHA1
#include <openssl/sha.h>

extern spe_program_handle_t spe_sha1_elf;

static int cell_sha1_init(EVP_MD_CTX *ctx)
	{
	DPRINTF("\n");

	return cell_digest_init(ctx, &spe_sha1_elf);
	}

static const EVP_MD cell_sha1 =
	{
	NID_sha1,
	NID_sha1WithRSAEncryption,
	SHA_DIGEST_LENGTH,
	0,
	cell_sha1_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	SHA_CBLOCK,
	sizeof(cell_md_ctx_t),
	};
#endif /* !OPENSSL_NO_SHA1 */

/* SHA224 / SHA256 */
#ifndef OPENSSL_NO_SHA256
#include <openssl/sha.h>

extern spe_program_handle_t spe_sha256_elf;

static int cell_sha256_init(EVP_MD_CTX *ctx)
	{
	DPRINTF("\n");

	return cell_digest_init(ctx, &spe_sha256_elf);
	}

static const EVP_MD cell_sha224 =
	{
	NID_sha224,
	NID_sha224WithRSAEncryption,
	SHA224_DIGEST_LENGTH,
	0,
	cell_sha256_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	SHA256_CBLOCK,
	sizeof(cell_md_ctx_t),
	};

static const EVP_MD cell_sha256 =
	{
	NID_sha256,
	NID_sha256WithRSAEncryption,
	SHA256_DIGEST_LENGTH,
	0,
	cell_sha256_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	SHA256_CBLOCK,
	sizeof(cell_md_ctx_t),
	};
#endif /* !OPENSSL_NO_SHA256 */

/* SHA384 / SHA512 */
#ifndef OPENSSL_NO_SHA512
#include <openssl/sha.h>

extern spe_program_handle_t spe_sha512_elf;

static int cell_sha512_init(EVP_MD_CTX *ctx)
	{
	DPRINTF("\n");

	return cell_digest_init(ctx, &spe_sha512_elf);
	}

static const EVP_MD cell_sha384 =
	{
	NID_sha384,
	NID_sha384WithRSAEncryption,
	SHA384_DIGEST_LENGTH,
	0,
	cell_sha512_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	SHA512_CBLOCK,
	sizeof(cell_md_ctx_t),
	};

static const EVP_MD cell_sha512 =
	{
	NID_sha512,
	NID_sha512WithRSAEncryption,
	SHA512_DIGEST_LENGTH,
	0,
	cell_sha512_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	SHA512_CBLOCK,
	sizeof(cell_md_ctx_t),
	};
#endif /* !OPENSSL_NO_SHA512 */

/* RIPEMD */
#ifndef OPENSSL_NO_RIPEMD
#include <openssl/ripemd.h>

extern spe_program_handle_t spe_ripemd_elf;

static int cell_ripemd_init(EVP_MD_CTX *ctx)
	{
	DPRINTF("\n");

	return cell_digest_init(ctx, &spe_ripemd_elf);
	}

static const EVP_MD cell_ripemd160 =
	{
	NID_ripemd160,
	NID_ripemd160WithRSA,
	RIPEMD160_DIGEST_LENGTH,
	0,
	cell_ripemd_init,
	cell_digest_update,
	cell_digest_final,
	cell_digest_copy,
	cell_digest_cleanup,
	EVP_PKEY_RSA_method,
	RIPEMD160_CBLOCK,
	sizeof(cell_md_ctx_t),
	};
#endif /* !OPENSSL_NO_RIPEMD */

/*** API function ***/

#define NUM_DIGESTS (sizeof(cell_digests_table) / sizeof(cell_digests_table[0]))

static const EVP_MD *cell_digests_table[] =
	{
#ifndef OPENSSL_NO_MD2
	&cell_md2,
#endif
#ifndef OPENSSL_NO_MD4
	&cell_md4,
#endif
#ifndef OPENSSL_NO_MD5
	&cell_md5,
#endif
#if !defined(OPENSSL_NO_SHA) && !defined(OPENSSL_NO_SHA0)
	&cell_sha,
#endif
#ifndef OPENSSL_NO_SHA1
	&cell_sha1,
#endif
#ifndef OPENSSL_NO_SHA256
	&cell_sha224,
	&cell_sha256,
#endif
#ifndef OPENSSL_NO_SHA512
	&cell_sha384,
	&cell_sha512,
#endif
#ifndef OPENSSL_NO_RIPEMD
	&cell_ripemd160,
#endif
	};

static int cell_digests_nids[NUM_DIGESTS];

static void cell_digests_global_init(void)
	{
	int i;
	static int init_p = 0;

	CRYPTO_w_lock(CRYPTO_LOCK_ENGINE);

	if (!init_p)
		{
		for (i = 0; i < NUM_DIGESTS; i++)
			{
			cell_digests_nids[i] = cell_digests_table[i]->type;
			}
		init_p = 1;
		}

	CRYPTO_w_unlock(CRYPTO_LOCK_ENGINE);
	}

int cell_digests(ENGINE *e, const EVP_MD **digest, const int **nids, int nid)
	{
	DPRINTF("EVP_MD: %p\n", digest);
	DPRINTF("NIDS: %p\n", nids);
	DPRINTF("NID: %d (%s)\n", nid, OBJ_nid2sn(nid));

	if (nid == 0)
		{
		cell_digests_global_init();

		*nids = cell_digests_nids;
		return NUM_DIGESTS;
		}
	else
		{
		int i;
		for (i = 0; i < NUM_DIGESTS; i++)
			{
			if (nid == cell_digests_table[i]->type)
				{
				*digest = cell_digests_table[i];
				return 1;
				}
			}
		CELLerr(CELL_F_CELL_DIGESTS,CELL_R_DIGEST_NOT_IMPLEMENTED);
		return 0;
		}
	}
