/*
 * salinfo.c - program to retrieve, log, and decode SAL error records
 *
 * Copyright (c) 2003 Hewlett-Packard Co
 *	Bjorn Helgaas <bjorn.helgaas@hp.com>
 *
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 *
 * ~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~~
 */

#include <sys/types.h>
#include <sys/poll.h>
#include <sys/stat.h>
#include <dirent.h>
#include <fcntl.h>
#include <stdio.h>
#include <stdlib.h>
#include <time.h>
#include <unistd.h>

#define ARRAY_SIZE(x)	(sizeof(x) / sizeof((x)[0]))

char *log_name[] = {
	"mca",
	"init",
	"cmc",
	"cpe",
};

struct salinfo_file {
	struct dirent *dirent;
	char *log_type;
	int fd;
	char *path;
};

int nfiles;
struct salinfo_file *salinfo_files;
struct pollfd *poll_table;

char *
salinfo_get_buffer(struct salinfo_file *s, int *psize)
{
	int nbytes, size, alloc;
	char *buffer;

	lseek(s->fd, 0, SEEK_SET);

	buffer = NULL;
	alloc = 16 * 1024;	// total buffer size
	size = 0;		// amount of buffer used so far
	do {
		buffer = realloc(buffer, alloc);
		if (!buffer) {
			fprintf(stderr, "Can't alloc %d bytes\n", alloc);
			exit(1);
		}

		nbytes = read(s->fd, buffer + size, alloc - size);
		if (nbytes == -1) {
			perror("read");
			exit(1);
		}

		if (nbytes == alloc - size)
			alloc *= 2;

		size += nbytes;
	} while (nbytes);

	if (size) {
		*psize = size;
		return buffer;
	}

	free(buffer);
	*psize = 0;
	return NULL;
}

void
salinfo_log(struct salinfo_file *s, char *buffer, int size)
{
	time_t t;
	struct tm tm;
	char path[64];
	int fd;

	if (!size)
		return;

	if (time(&t) == (time_t) -1) {
		perror("time");
		exit(1);
	}

	gmtime_r(&t, &tm);

	sprintf(path, "%04d_%02d_%02d_%02d_%02d_%02d_%s_%s", 
		1900 + tm.tm_year, tm.tm_mon + 1, tm.tm_mday,
		tm.tm_hour, tm.tm_min, tm.tm_sec,
		s->dirent->d_name, s->log_type);
	fd = creat(path, S_IRUSR | S_IRGRP | S_IROTH);
	if (fd == -1) {
		perror(path);
		exit(1);
	}

	write(fd, buffer, size);
	fsync(fd);
	close(fd);
}

void
salinfo_clear_log(struct salinfo_file *s)
{
	lseek(s->fd, 0, SEEK_SET);
	write(s->fd, "clear", 5);
}

void
salinfo_log_event(struct salinfo_file *s)
{
	char *buffer;
	int size;

	buffer = salinfo_get_buffer(s, &size);
	if (!buffer)
		return;

	salinfo_log(s, buffer, size);

	printf("+BEGIN HARDWARE ERROR STATE from %s\n", s->path);
	ia64_log_platform_info_print(buffer, printf);
	printf("+END HARDWARE ERROR STATE from %s\n", s->path);

	free(buffer);
	salinfo_clear_log(s);
}

int
select_cpu_dir(const struct dirent *d)
{
	if (!strncmp(d->d_name, "cpu", 3))
		return 1;
	return 0;
}

void
open_files(char *dir)
{
	struct dirent **namelist;
	struct pollfd *p;
	struct salinfo_file *s;
	char *path;
	int i, j, n, fd;

	n = scandir(dir, &namelist, select_cpu_dir, alphasort);
	if (n == -1) {
		perror("scandir");
		exit(1);
	}

	nfiles = n * ARRAY_SIZE(log_name);
	poll_table = malloc(nfiles * sizeof(*poll_table));
	if (!poll_table) {
		fprintf(stderr, "Can't alloc poll table\n");
		exit(1);
	}

	salinfo_files = malloc(nfiles * sizeof(*salinfo_files));
	if (!salinfo_files) {
		fprintf(stderr, "Can't alloc salinfo file table\n");
		exit(1);
	}

	p = poll_table;
	s = salinfo_files;
	for (i = 0; i < n; i++) {
		for (j = 0; j < ARRAY_SIZE(log_name); j++) {
			path = malloc(strlen(dir) + strlen(namelist[0]->d_name) + 4 + 2 + 4);
			if (!path) {
				fprintf(stderr, "Can't alloc path\n");
				exit(1);
			}

			sprintf(path, "%s/%s/%s", dir, namelist[i]->d_name, log_name[j]);
			fd = open(path, O_RDWR);
			if (fd == -1) {
				perror(path);
				exit(1);
			}

			p->fd = fd;
			p->events = POLLIN;
			p++;

			s->dirent = namelist[i];
			s->log_type = log_name[j];
			s->path = path;
			s->fd = fd;
			s++;
		}
	}
}

main(int argc, char **argv)
{
	int args = argc - 1;
	int i, rc;

	if (args != 1) {
		fprintf(stderr, "Usage: %s <dir>\n", argv[0]);
		exit(1);
	}

	open_files(argv[1]);

	while (1) {
		rc = poll(poll_table, nfiles, -1);
		if (rc == -1) {
			perror("poll");
			exit(1);
		}

		for (i = 0; i < nfiles; i++)
			if (poll_table[i].revents & POLLIN)
				salinfo_log_event(&salinfo_files[i]);
	}

	exit(0);
}
