/* -*- linux-c -*- ------------------------------------------------------- *
 *   
 *   Copyright 2002 H. Peter Anvin - All Rights Reserved
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation, Inc., 53 Temple Place Ste 330,
 *   Bostom MA 02111-1307, USA; either version 2 of the License, or
 *   (at your option) any later version; incorporated herein by reference.
 *
 * ----------------------------------------------------------------------- */

/*
 * raid6algos.c
 *
 * Algorithm list and algorithm selection for RAID-6
 */

#include "raid6.h"
#ifndef __KERNEL__
#include <sys/mman.h>
#endif

/* Various routine sets */
extern const struct raid6_calls raid6_intx1;
extern const struct raid6_calls raid6_intx2;
extern const struct raid6_calls raid6_intx4;
extern const struct raid6_calls raid6_intx8;
extern const struct raid6_calls raid6_intx16;
#if defined(__i386__) || defined(__x86_64__)
extern const struct raid6_calls raid6_mmxx1;
extern const struct raid6_calls raid6_mmxx2;
extern const struct raid6_calls raid6_sse1x1;
extern const struct raid6_calls raid6_sse1x2;
extern const struct raid6_calls raid6_sse2x1;
extern const struct raid6_calls raid6_sse2x2;
#endif

const struct raid6_calls * const raid6_algos[] = {
	&raid6_intx1,
	&raid6_intx2,
	&raid6_intx4,
	&raid6_intx8,
	&raid6_intx16,
#if defined(__i386__) || defined(__x86_64__)
	&raid6_mmxx1,
	&raid6_mmxx2,
	&raid6_sse1x1,
	&raid6_sse1x2,
	&raid6_sse2x1,
	&raid6_sse2x2,
#endif
	NULL
};



/* Try to pick the best algorithm */
/* This code uses the gfmul table as convenient data set to abuse */

#ifndef PAGE_SIZE
#define PAGE_SIZE 4096
#endif

void raid6_select_algo(void)
{
	const struct raid6_calls * const * algo;
	const struct raid6_calls * best;
	void *dptrs[(65536/PAGE_SIZE)+2];
	int i, disks;
	unsigned long perf, bestperf;
	int bestprefer;
	unsigned long j0, j1;

	disks = (65536/PAGE_SIZE)+2;
	for ( i = 0 ; i < disks-2 ; i++ ) {
		dptrs[i] = ((char *)raid6_gfmul) + PAGE_SIZE*i;
	}
	/**** FIX: ALLOCATE PROPERLY FOR THE KERNEL ****/
	{
		char *syndromes;
		syndromes = mmap(NULL, 2*PAGE_SIZE, PROT_READ|PROT_WRITE,
			 MAP_PRIVATE|MAP_ANONYMOUS, 0, 0);
		dptrs[disks-2] = syndromes;
		dptrs[disks-1] = syndromes + PAGE_SIZE;
	}

	bestperf = 0;  bestprefer = 0;  best = NULL;

	for ( algo = raid6_algos ; *algo ; algo++ ) {
		if ( !(*algo)->valid || (*algo)->valid() ) {
			perf = 0;
			j0 = raid6_jiffies();
			while ( (j1 = raid6_jiffies()) == j0 )
				cpu_relax();
			while ( (raid6_jiffies() - j1) < 16 ) {
				(*algo)->gen_syndrome(disks, PAGE_SIZE, dptrs);
				perf++;
			}
			if ( (*algo)->prefer > bestprefer ||
			     (*algo)->prefer == bestprefer && perf > bestperf ) {
				best = *algo;
				bestprefer = best->prefer;
				bestperf = perf;
			}
			printk("raid6: %-8s %5d MB/s\n", (*algo)->name,
			       (perf << (16-4))/HZ);
		}
	}

	printk("raid6: using algorithm %s\n", best->name);
	raid6_call = *best;
}

