#define UwsCmd_c


// **********************************************************************
// * uwscmd.c                                                           *
// *                                                                    *
// * Written by Matthew DeLoera (deloera@us.ibm.com)                    *
// * (C) Copyright IBM Corporation, 1998-2001                           *
// *                                                                    *
// *  This software may be used and distributed according to the terms  *
// *  of the GNU Public License, incorporated herein by reference.      *
// **********************************************************************

// **********************************************************************
// * uwscmd.c - This source file implements all primary driver entry    *
// *            points, and is thought of as the "main" source file for *
// *            the IBM Advanced System Management Processor device     *
// *            driver.                                                 *
// *                                                                    *
// * Exported Functions                                                 *
// * ------------------                                                 *
// * init_module    - standard module loading routine                   *
// * cleanup_module - standard module unloading routine                 *
// * sysSp_ioctl    - IOCtl entry point                                 *
// * sysSp_open     - driver open entry point                           *
// * sysSp_close    - driver close entry point                          *
// *                                                                    *
// * Changes                                                            *
// * -------                                                            *
// * 01/29/2001 - Cleaned up for open source release.                   *
// * ----------------------------------------------------------------   *
// * 05/14/2001 - PHR_161788 - Implemented usage count macro so the     *
// *              device driver cannot be removed while it's busy       *
// * ----------------------------------------------------------------   *
// * 06/27/2002 - Added multi node support       // PHR_MULTI_4         *
// *                                                                    *
// **********************************************************************


// ************
// * INCLUDES *
// ************

#include "uwscmd.h"


// ***********************
// * EXTERNAL PROTOTYPES *
// ***********************

extern int  sysSpinit(int IRQoverride);
#ifndef IA64                              // PHR_180930
  extern void sysSpASRshutdown(void);
#endif                                    // PHR_180930 
extern void sysSpPurgeQueueAll(PDRIVER_INFO pDriverNode, void *pHead);
extern void sysSpDisplayQueue(void *pHead);
extern void sysSpDisableInterrupts(PDRIVER_INFO pDriverNode, unsigned long templong);
extern void RemoveNode(PDRIVER_INFO pDriverNode);
extern int sysSpProcessIoctl(volatile int cmd, volatile PDD_IOCTL_INFO pDDIoctlInfo, volatile unsigned int PID, volatile int callerFlag, PDRIVER_INFO pDriverNode);
extern void sysSpPurgeQueueByPID(PDRIVER_INFO pDriverNode, void *pHead, unsigned int PID);


// ********************
// * LOCAL PROTOTYPES *
// ********************
int init_module(void);
void cleanup_module(void);
int sysSp_ioctl(struct inode *pInode,struct file *pFile, unsigned int uiArg1, unsigned long ulArg2);
int sysSp_open(struct inode *pInode,struct file *pFile);
int sysSp_close(struct inode *pInode, struct file *pFile);


// ************************************************
// * GLOBALS FOR COMMUNICATION WITH SHARED DRIVER *
// ************************************************
// Start of PHR_198187 
extern int           NumSpNodes;
extern PDRIVER_INFO  pSpNodeArray;        // ARRAY OF PDRIVER_INFO STRUCTS
extern PDRIVER_INFO  pSpPrimaryNode;      // PRIMARY NODE POINTER            
// End of PHR_198187 

unsigned char IBMASMibmasmexists = 0;

// ****************************************************************************
// * init_module() - Well-defined entry point that gets called when insmod is *
// *                 executed on the druver module. It first registers a /dev *
// *                 emtry, then calls sysSpinit() to do all device and       *
// *                 driver initialization and startup.                       *
// ****************************************************************************
int init_module(void)
{
     int  iResult, hide = 0;
     if (DBG_FUNCENTEXT) printk(KERN_CRIT "init_module: Entry.\n");

     // THIS BLOCK ELIMINATES COMPILER WARNINGS FOR UNUSED VARIABLES
     hide = DBG_PCIDEBUG;    hide = DBG_TIMER;     hide = DBG_BLOCKS;
     hide = DBG_HB;          hide = DBG_DEBUG;     hide = DBG_MULTI;
     hide = DBG_QUEUES;      hide = DBG_INIT;      hide = DBG_EVENTS;
     hide = DBG_IRQ;         hide = DBG_ISADEBUG;  hide = DBG_SLIM;
     hide = DBG_REV_HB;      hide = DBG_FORCEASR;  hide = DBG_DDINST;
     hide = DBG_ASR;         hide = DBG_ASR_TEST;  hide = DBG_SIGNALS;
     hide = DBG_LOCKS;       hide = DBG_PHR;       hide = DBG_MOUSE;
     hide = DBG_CMDTIMEOUT;

     // IF THE CALLER DOES NOT HAVE THE RIGHT PERMISSIONS, REJECT IOCTL REQUESTS
     if (!capable(CAP_SYS_ADMIN))
     {
        if (DBG_IOCTL)
        {
           printk(KERN_CRIT "ibmasm: DBG_IOCTL - Caller does not have correct permissions. Ioctl request rejected.\n");
        }

        return -EPERM;
     }

     // ANNOUNCE NO-SHUTDOWN MODE
     if (DBG_NOSHUTDOWN)
        printk(KERN_CRIT "init_module: No OS Shutdown command will be sent.\n");

     // REGISTER THE DEV ENTRY
     iResult = register_chrdev(iSysSP_Major,"ibmasm", &sysSp_fops);
     if (iResult < 0) 
     {
          printk(KERN_CRIT "init_module: Unable to get major device number for ibmasm. Aborting.\n");
          return iResult;
     }
     iSysSP_Major = iResult;

     // ATTEMPT DRIVER AND SP CONFIGURATION
     if ( (iResult = sysSpinit(DBG_IRQOVERRIDE)) )
     {
          unregister_chrdev(iSysSP_Major,"ibmasm");
          printk(KERN_CRIT "init_module: ibmasm driver initialization failed. Aborting.\n");
          return iResult;
     }

#ifdef __SMP__
     if ( pSpPrimaryNode->AsmType == ASMTYPE_ASR    ||
          pSpPrimaryNode->AsmType == ASMTYPE_JASPER || 
          pSpPrimaryNode->AsmType == ASMTYPE_PEARL   )
     {
         printk(KERN_CRIT "IBM Advanced System Management SMP Device Driver loaded (ASR).\n");
     }
     else
     {
         // DRIVER SUCCESSFULLY LOADED - COMMUNICATE THIS TO SHARED COM DRIVER
         IBMASMibmasmexists = 1;

         printk(KERN_CRIT "IBM Advanced System Management SMP Device Driver loaded.\n");
     }
#else
     if ( pSpPrimaryNode->AsmType == ASMTYPE_ASR    || 
          pSpPrimaryNode->AsmType == ASMTYPE_JASPER || 
          pSpPrimaryNode->AsmType == ASMTYPE_PEARL   )
     {
         printk(KERN_CRIT "IBM Advanced System Management Device Driver loaded (ASR).\n");
     }
     else
     {
         // DRIVER SUCCESSFULLY LOADED - COMMUNICATE THIS TO SHARED COM DRIVER
         IBMASMibmasmexists = 1;

         printk(KERN_CRIT "IBM Advanced System Management Device Driver loaded.\n");
     }

     // End of PHR_MULTI_1
#endif

     if (DBG_FUNCENTEXT) printk(KERN_CRIT "init_module: Exit.\n");
     return 0;
}


// ******************************************************************************
// * cleanup_module() - Well-defined entry point that gets called when rmmod is *
// *                    executed on the druver module. It allows all currently  *
// *                    executing commands to complete, stops the heartbeat     *
// *                    process if necessary, purges the command/event queues,  *
// *                    releases the interrupt and /dev devices, and exits.     *
// *                                                                            *
// * NOTE - In ASR mode, we follow an alternate shutdown path.                  *
// ******************************************************************************
void cleanup_module(void)
{
   PDRIVER_INFO    pDriverNode;
   DD_IOCTL_INFO   dd_Ioctl_Info;
   PDD_IOCTL_INFO  pdd_Ioctl_Info;
   UCHAR          *pCmdBuffer;
   ULONG           flags;

   int  rc;
   int  NodeIndex; 
   u32  tempUlong;   


   //ULONG tempUlong = 0;             // PHR_180930

   // INITIALIZATION
   ShuttingDown   = 1;
   pDriverNode    = NULL;
   pdd_Ioctl_Info = &dd_Ioctl_Info;
   rc             = RC_SUCCESS;
   flags          = 0;
   tempUlong      = 0;
   NodeIndex      = 0;

   // IF THE CALLER DOES NOT HAVE THE RIGHT PERMISSIONS, REJECT IOCTL REQUESTS
   if (!capable(CAP_SYS_ADMIN))
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Caller does not have correct permissions. Ioctl request rejected.\n");
      }

      return;
   }

   #ifndef IA64                                          // PHR_180930 
   // FOLLOW ALTERNATE SHUTDOWN PATH FOR ASR MODE
   if ( pSpPrimaryNode->AsmType == ASMTYPE_ASR    ||
        pSpPrimaryNode->AsmType == ASMTYPE_JASPER || 
        pSpPrimaryNode->AsmType == ASMTYPE_PEARL   )
   {
        if (DBG_SHUTDOWN)
        {
           printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN - Shutting down ASR driver.\n");
        }

        // DO ASR SHUTDOWN
        sysSpASRshutdown();

        // UN-REGISTER THE /dev DEVICE
        unregister_chrdev(iSysSP_Major,"ibmasm");

        return;
   }
   #endif
                                                    // PHR_180930 
   if (DBG_SHUTDOWN)
   {
      printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN - Shutting down device driver.\n");
   }

   // ALLOCATE AND INITIALIZE THE COMMAND BUFFER
   pCmdBuffer = NULL;
   pCmdBuffer = (unsigned char *)kmalloc((SEND_BUFFER_SIZE), GFP_KERNEL);

   if ( !pCmdBuffer  )
   {
      printk(KERN_CRIT "ibmasm: ERROR - cleanup_module failed to allocate command buffer. Exiting.\n");
   }
   
   // CLEANUP EACH NODE
   for ( NodeIndex = NumSpNodes-1; NodeIndex >= 0; NodeIndex--)
   {
      pDriverNode = pSpNodeArray + NodeIndex;

      // PURGE ALL EXTERNALLY-GENERATED COMMANDS 
      if (DBG_SHUTDOWN)
      {
         printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN - Purging all queued commands for SP node %i.\n", NodeIndex);
      }

      spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );
      sysSpPurgeQueueAll(pDriverNode, pDriverNode->CmdQueue);
      spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );


      // PURGE ALL EVENTS 
      if (DBG_SHUTDOWN)
      {
         printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN - Purging all queued events for SP node %i.\n", NodeIndex);
      }

      spin_lock_irqsave( &(pDriverNode->EventQueueLock), flags );
      sysSpPurgeQueueAll(pDriverNode, pDriverNode->EventQueue);
      spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );

      // SEND THE 4.3.6 (OS IS DOWN) COMMAND TO THE SP
      if (pCmdBuffer)
      {
         memset(pCmdBuffer, 0, SEND_BUFFER_SIZE );

         if (DBG_SHUTDOWN)
         {
             printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN - cleanup_module - Sending 4.3.6 to SP....\n");
         }

         pCmdBuffer[0] = spWrite;  // Command Type   
         pCmdBuffer[1] = 3;        // Command Length 
         pCmdBuffer[2] = 1;        // Data Length  LSB  
         pCmdBuffer[3] = 0;        // Data Length  MSB         
         pCmdBuffer[4] = 0;        // Status         
         pCmdBuffer[5] = 0;        // Reserved       
         pCmdBuffer[6] = 4;        // Command Byte 0 
         pCmdBuffer[7] = 3;        // Command Byte 1 
         pCmdBuffer[8] = 6;        // Command Byte 2 
         pCmdBuffer[9] = 4;        // Data Byte 0    

         // INITIALIZE DEVICE DRIVER STRUCTURE INTERFACE
         dd_Ioctl_Info.returnCode   = 0;
         dd_Ioctl_Info.bufferLength = 10;
         dd_Ioctl_Info.pBuffer      = pCmdBuffer;

         rc = sysSpProcessIoctl(IOCTL_RW, pdd_Ioctl_Info, PID_OS_IS_UP, INTERNAL_IOCTL, pDriverNode);

         if (rc == 0)
         {
            printk(KERN_CRIT "ibmasm: ERROR - 4.3.6 (OS IS DOWN) failed\n");
         }
      }

      // DISABLE COMMAND TIMER
      if (DBG_SHUTDOWN)
      {
         printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN - Deleting the command timer.\n");
      }

      del_timer(&(pDriverNode->CommandTimer));

      // IF WISEMAN/EAGLE, DISABLE SP INTERRUPTS
      if ( (pDriverNode->AsmType == ASMTYPE_WISEMAN) || 
           (pDriverNode->AsmType == ASMTYPE_EAGLE)    )
      {
         sysSpDisableInterrupts(pDriverNode, tempUlong);
      }

      if (DBG_SHUTDOWN)
      {
         printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN - Removing node %i.\n", NodeIndex);
      }

      RemoveNode(pDriverNode);
      
   }  // end for NumSpNodes

   if (pCmdBuffer)
   {
      kfree(pCmdBuffer);
   }

   // UN-REGISTER THE /dev DEVICE
   unregister_chrdev(iSysSP_Major,"ibmasm");      // PHR_MULTI_TODO

   if (DBG_SHUTDOWN)
   {
       printk(KERN_CRIT "ibmasm: DBG_SHUTDOWN - sysSp driver unloaded.\n");  
   }

   return;
}

// *******************************************************************************
// * sysSp_ioctl() - Registered IOCtl entry point. All requests passing through  *
// *                 this routine are coming from the outside world. We first    *
// *                 ensure that the driver is not unloading. We then fetch the  *
// *                 associated PID and pass the request to sysSpProcessIoctl(). *
// *                                                                             *
// * NOTE - In ASR mode, we reject all IOCtl requests. In other words, all       *
// *        normally-supported IOCtl's become unknown.                           *
// *                                                                             *
// * Parameters:                                                                 *
// *      pInode - file info struct associated with the opened dev node          *
// *      pFile  - file info struct associated with the calling application      *
// *      uiArg1 - IOCtl command code                                            *
// *      uiArg2 - pointer to the IOCtl parm structure (typecast when using)     *
// *                                                                             *
// * Return Codes:                                                               *
// *      0 = failed  - check parm structure for specific error code             *
// *      1 = success                                                            *
// *******************************************************************************
int sysSp_ioctl( struct inode  *pInode,
                 struct file   *pFile,
                 unsigned int   uiArg1,
                 unsigned long  ulArg2 )
{
   int rc;
   unsigned int *pPID;
   unsigned long flags;
   unsigned int minor;

   PDRIVER_INFO    pDriverNode;
   PDD_IOCTL_INFO  pDDIoctlInfo;

   flags = 0;

   // IF THE CALLER DOES NOT HAVE THE ROOT ACCESS, REJECT IOCTL REQUESTS
   if (!capable(CAP_SYS_ADMIN))
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Caller does not have correct permissions. Ioctl request rejected.\n");
      }

      return -EPERM;
   }

   // CHECK THE ARGUMENTS PASSED TO THE DEVICE DRIVER 
   if ( !ulArg2 || !pFile || !pInode )
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - sysSp_Ioctl was passed an invald parm. Ioctl request rejected.\n");
      }

      return -EINVAL;
   }
   else
   {
       pDDIoctlInfo = (PDD_IOCTL_INFO)ulArg2;

       // VERIFY WRITE ACCESS TO THE BUFFER THE CALLER HAS PASSED
       rc = !access_ok(VERIFY_WRITE, pDDIoctlInfo->pBuffer, pDDIoctlInfo->bufferLength ); 

       if (rc)
       {
          if (DBG_IOCTL)
          {
             printk(KERN_CRIT "ibmasm: DBG_IOCTL - sysSp_Ioctl was passed an invald buffer. Ioctl request rejected.\n");
          }
          return -EFAULT;
       }
   }

   // IF DRIVER IS UNLOADING, REJECT IOCTL REQUESTS
   if ( ShuttingDown )
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Driver unloading. Ioctl request rejected.\n");
      }

      pDDIoctlInfo->returnCode = DDERR_DRIVER_CLOSED;

      return -ENODEV;
   }

   // IF DRIVER IS IN ASR MODE, REJECT IOCTL REQUESTS
   if( pSpPrimaryNode->AsmType == ASMTYPE_ASR    || 
       pSpPrimaryNode->AsmType == ASMTYPE_JASPER || 
       pSpPrimaryNode->AsmType == ASMTYPE_PEARL   )
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - ASR mode active - Ioctl request rejected..\n");
      }

      pDDIoctlInfo->returnCode = DDERR_UNKNOWN_IOCTL;

      return -EINVAL;
   }

   if (pSpPrimaryNode->AsmType == ASMTYPE_EAGLE)
   {
       minor = MINOR(pInode->i_rdev);     
       // VERIFY MINOR NUMBER AND SET pDriverNode TO CORRECT NODE
       if (minor >= NumSpNodes)
       {
          if (DBG_IOCTL)
          {
             printk(KERN_CRIT "ibmasm: DBG_IOCTL - Invalid minor - Ioctl request rejected..\n");
          }

          return -ENODEV;                 
       }

       pDriverNode = pSpNodeArray + minor;       
   }
   else
   {
       pDriverNode = pSpPrimaryNode;       
   }

   // FETCH THIS APP'S PID FROM THE GLOBAL PID TABLE
   pPID = (unsigned int *)(pFile->private_data);

   if (DBG_IOCTL)
   {
       printk(KERN_CRIT "ibmasm: DBG_IOCTL - Received IOCTL from PID %x.\n", *pPID);
   }

   rc = sysSpProcessIoctl(uiArg1, pDDIoctlInfo, *pPID, EXTERNAL_IOCTL, pDriverNode);

   return rc;
}


// ********************************************************************************
// * sysSp_open() - Well-defined entry point that gets called when an application *
// *                opens this driver. Ensures device is present, and registers   *
// *                the calling application's PID in the global PID table.        *
// *                                                                              *
// * NOTE - In ASR mode, I don't want any client apps to be able to open the      *
// *        driver, since I don't support any IOCtl's in that mode. So, in ASR    *
// *        mode, always fail by indicating no device.                            *
// *                                                                              *
// * Parameters:                                                                  *
// *      pInode                                                                  *
// *      pFile                                                                   *
// ********************************************************************************
int sysSp_open(struct inode *pInode,struct file *pFile)
{
   UINT   i;
   UINT   minor;
   
   PDRIVER_INFO pDriverNode;

   if (DBG_IOCTL)
   {
      printk(KERN_CRIT "ibmasm: DBG_IOCTL - Entered sysSp_open.\n");
   }

   // IF THE CALLER DOES NOT HAVE THE RIGHT PERMISSIONS, REJECT IOCTL REQUESTS
   if (!capable(CAP_SYS_ADMIN))
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Caller does not have correct permissions. Ioctl request rejected.\n");
      }

      return -EPERM;
   }

   // CHECK INPUT PARMS
   if ( !pInode || !pFile )
   {
       if (DBG_IOCTL)
       {
          printk(KERN_CRIT "ibmasm: DBG_IOCTL - sysSp_Ioctl was passed an invald parm. Ioctl request rejected.\n");
       }

       return -EINVAL;
   }

   // ENSURE WE'RE NOT IN ASR MODE
   if(pSpPrimaryNode->AsmType == ASMTYPE_ASR    ||
      pSpPrimaryNode->AsmType == ASMTYPE_PEARL  ||
      pSpPrimaryNode->AsmType == ASMTYPE_JASPER )              
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Opening the SP ASM driver is invalid when in ASR mode.\n");
      }

      return -ENODEV;
   }

   if (pSpPrimaryNode->AsmType == ASMTYPE_EAGLE )
   {
       minor = MINOR( pInode->i_rdev );

       // VERIFY MINOR NUMBER 
       if ( minor >= NumSpNodes )
       {
          if (DBG_IOCTL)
          {
             printk(KERN_CRIT "ibmasm: DBG_IOCTL - Invalid minor, minor is > number of valid nodes.\n");
          }

          return -ENODEV;   
       }

       // SET UP THIS NODES PLACE IN THE NODE ARRAY
       pDriverNode = pSpNodeArray + minor;               
   }
   else
   {
      pDriverNode= pSpPrimaryNode;
   }


   // ENSURE DEVICE IS PRESENT
   if ( pDriverNode->devPresent == FALSE )
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Device not present. Aborting open.\n");
      }

      return -ENODEV;
   }

   // LOCATE AN AVAILABLE PID TABLE SLOT AND REGISTER THIS PID
   for ( i = 0; i < MAXPIDS; i++ )
   {
      if (!pDriverNode->PIDtable[i].valid)
      {
         break;
      }
   }

   // FAIL OPEN IF PID TABLE MAXED OUT 
   if ( i == MAXPIDS )
   {
      if (DBG_IOCTL)
      {
        printk(KERN_CRIT "ibmasm: DBG_IOCTL - PID table full. Aborting open.\n");
      }

      return -ENODEV;
   }

   pDriverNode->PIDtable[i].PID   = current->pid;
   pDriverNode->PIDtable[i].valid = 1;

   if (DBG_IOCTL)
   {
     printk(KERN_CRIT "ibmasm: DBG_IOCTL - Registered PID %x.\n", current->pid);
   }

   // CACHE PID BACK INTO pFile->private_data
   pFile->private_data = (void *)&(pDriverNode->PIDtable[i].PID); 

   // INCREMENT THE USE COUNT
   MOD_INC_USE_COUNT;    

   return 0;
}


// *********************************************************************************
// * sysSp_close() - Well-defined entry point that gets called when an application *
// *                 closes this driver. Ensures command and event queues are      *
// *                 purged of entries for this particular PID.                    *
// *                                                                               *
// * NOTE - In ASR mode, I don't want any client apps to be able to close the      *
// *        driver, since I don't support any IOCtl's in that mode. So, in ASR     *
// *        mode, always fail. Of course, they couldn't have opened the driver     *
// *        either, but you know how it is...... :)     Simply indicate no device. *
// *                                                                               *
// * Parameters:                                                                   *
// *      pInode                                                                   *
// *      pFile                                                                    *
// *********************************************************************************
int sysSp_close(struct inode *pInode, struct file *pFile)
{
   UINT  i;
   UINT  minor;
   ULONG flags;

   PDRIVER_INFO pDriverNode;

   minor = 0;

   // IF THE CALLER DOES NOT HAVE THE RIGHT PERMISSIONS, REJECT IOCTL REQUESTS
   if (!capable(CAP_SYS_ADMIN))
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Caller does not have correct permissions. Ioctl request rejected.\n");
      }

      return -EPERM;
   }

   // CHECK INPUT PARMS
   if ( !pInode || ! pFile )
   {
       if (DBG_IOCTL)
       {
          printk(KERN_CRIT "ibmasm: DBG_IOCTL - sysSp_Ioctl was passed an invald parm. Ioctl request rejected.\n");
       }

       return -EINVAL;
   }

   // ENSURE WE'RE NOT IN ASR MODE
   if ( pSpPrimaryNode->AsmType == ASMTYPE_ASR     ||
        pSpPrimaryNode->AsmType == ASMTYPE_PEARL   ||
        pSpPrimaryNode->AsmType == ASMTYPE_JASPER  )              
   {
      if (DBG_IOCTL)
      {
        printk(KERN_CRIT "ibmasm: DBG_IOCTL - Closing the SP ASM driver is invalid when in ASR mode.\n");
      }
      return -ENODEV;
   }

   flags = 0;

   if (pSpPrimaryNode->AsmType == ASMTYPE_EAGLE)
   {
       minor = MINOR(pInode->i_rdev);

       // VERIFY MINOR NUMBER 
       if ( minor >= NumSpNodes )
       {
          if (DBG_IOCTL)
          {
             printk(KERN_CRIT "ibmasm: DBG_IOCTL - Invalid minor, minor is > number of valid nodes.\n");
          }

          return -ENODEV; 
       }

       // SET UP THIS NODES PLACE IN THE NODE ARRAY
       pDriverNode = pSpNodeArray + minor;               
   }
   else
   {
       pDriverNode = pSpPrimaryNode;
   }


   // ENSURE DEVICE IS PRESENT
   if ( pDriverNode->devPresent == FALSE )
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - Device not present. Aborting clsoe.\n");
      }

      return -ENODEV;
   }

   // FIRST LOCATE THIS PID IN GLOBAL PID TABLE BEFORE CONTINUING
   for (i = 0; i < MAXPIDS; i++)
   {      
      if( (pDriverNode->PIDtable[i].PID == current->pid) && 
          (pDriverNode->PIDtable[i].valid)               )
      {
         if (DBG_IOCTL)
         {
            printk(KERN_CRIT "ibmasm: DBG_IOCTL - PID %x is closing device driver node %i.\n", current->pid, minor);
         }

         // RELEASE THIS PID FROM TABLE
         pDriverNode->PIDtable[i].valid = 0;

         if (DBG_IOCTL)
         {
            printk(KERN_CRIT "ibmasm: DBG_IOCTL - Releasing PID %x.\n", current->pid);
         }

         // PURGE ALL QUEUED COMMANDS FOR THIS PID
         if (DBG_IOCTL)
         {
            printk(KERN_CRIT "ibmasm: DBG_IOCTL - Purging command queue for PID %x.\n", current->pid);
         }

         spin_lock_irqsave( &(pDriverNode->CmdQueueLock), flags );

         sysSpPurgeQueueByPID(pDriverNode, pDriverNode->CmdQueue, current->pid);

         spin_unlock_irqrestore( &(pDriverNode->CmdQueueLock), flags );

         // PURGE ALL QUEUED EVENT REGISTRATIONS FOR THIS PID
         if (DBG_IOCTL)
         {
            printk(KERN_CRIT "ibmasm: DBG_IOCTL - Purging all registered events for PID %x.\n", current->pid);
         }

         spin_lock_irqsave( &(pDriverNode->EventQueueLock), flags );

         sysSpPurgeQueueByPID(pDriverNode, pDriverNode->EventQueue, current->pid);

         spin_unlock_irqrestore( &(pDriverNode->EventQueueLock), flags );

         break;
      }
   }

   if ( i == MAXPIDS )
   {
      if (DBG_IOCTL)
      {
         printk(KERN_CRIT "ibmasm: DBG_IOCTL - unable to locate PID in table. Aborting Close.\n");
      }
   }

   // DECREMENT USAGE COUNT
   MOD_DEC_USE_COUNT;     

   return 0;
}


#undef UwsCmd_c


