/* 
 * grsecurity/gracl_ip.c
 * Copyright Brad Spengler 2002, 2003
 *
 */

#include <linux/kernel.h>
#include <asm/uaccess.h>
#include <asm/errno.h>
#include <net/sock.h>
#include <linux/file.h>
#include <linux/fs.h>
#include <linux/net.h>
#include <linux/in.h>
#include <linux/skbuff.h>
#include <linux/ip.h>
#include <linux/udp.h>
#include <linux/smp_lock.h>
#include <linux/types.h>
#include <linux/sched.h>
#include <linux/gracl.h>
#include <linux/grsecurity.h>
#include <linux/grinternal.h>

#define GR_BIND 	0x01
#define GR_CONNECT 	0x02

int
gr_search_socket(const int domain, const int type, const int protocol)
{
	struct acl_subject_label *curr;

	if (unlikely(!gr_acl_is_enabled()))
		goto exit;

	if ((domain < 0) || (type < 0) || (protocol < 0) || (domain != PF_INET)
	    || (domain >= NPROTO) || (type >= SOCK_MAX) || (protocol > 255))
		goto exit;	// let the kernel handle it

	curr = current->acl;

	if (!curr->ips)
		goto exit;

	if ((curr->ip_type & (1 << type)) &&
	    (curr->ip_proto[protocol / 32] & (1 << (protocol % 32))))
		goto exit;

	if (curr->mode & GR_LEARN) {
		/* we don't place acls on raw sockets , and sometimes
		   dgram/ip sockets are opened for ioctl and not
		   bind/connect, so we'll fake a bind learn log */
		if (type == SOCK_RAW || type == SOCK_PACKET) {
			__u32 fakeip = 0;
			security_learn(GR_IP_LEARN_MSG, curr->device,
				       curr->inode, NIPQUAD(fakeip), 0, type,
				       protocol, GR_CONNECT);
		} else if ((type == SOCK_DGRAM) && (protocol == IPPROTO_IP)) {
			__u32 fakeip = 0;
			security_learn(GR_IP_LEARN_MSG, curr->device,
				       curr->inode, NIPQUAD(fakeip), 0, type,
				       protocol, GR_BIND);
		}
		/* we'll log when they use connect or bind */
		goto exit;
	}

	security_alert(GR_SOCK_MSG, domain, type, protocol, DEFAULTSECARGS);

	return 0;
      exit:
	return 1;
}

static __inline__ int
gr_search_connectbind(const int mode, const struct sock *sk,
		      const struct sockaddr_in *addr, const int type)
{
	struct acl_subject_label *curr;
	struct acl_ip_label *ip;
	unsigned long i;
	__u32 ip_addr = 0;
	__u16 ip_port = 0;

	if (unlikely(!gr_acl_is_enabled() || sk->family != PF_INET))
		return 1;

	curr = current->acl;

	if (!curr->ips)
		return 1;

	ip_addr = addr->sin_addr.s_addr;
	ip_port = ntohs(addr->sin_port);

	for (i = 0; i < curr->ip_num; i++) {
		ip = *(curr->ips + i);
		if ((ip->mode & mode) &&
		    (ip_port >= ip->low) &&
		    (ip_port <= ip->high) &&
		    ((ntohl(ip_addr) & ip->netmask) ==
		     (ntohl(ip->addr) & ip->netmask))
		    && (ip->
			proto[sk->protocol / 32] & (1 << (sk->protocol % 32)))
		    && (ip->type & (1 << type)))
			return 1;
	}

	if (curr->mode & GR_LEARN) {
		security_learn(GR_IP_LEARN_MSG, curr->device, curr->inode,
			       NIPQUAD(ip_addr), ip_port, type,
			       sk->protocol, mode);
		return 1;
	}

	if (mode == GR_BIND)
		security_alert(GR_BIND_ACL_MSG, NIPQUAD(ip_addr), ip_port, type,
			       sk->protocol, DEFAULTSECARGS);
	else if (mode == GR_CONNECT)
		security_alert(GR_CONNECT_ACL_MSG, NIPQUAD(ip_addr), ip_port,
			       type, sk->protocol, DEFAULTSECARGS);

	return 0;
}

int
gr_search_connect(const struct socket *sock, const struct sockaddr_in *addr)
{
	return gr_search_connectbind(GR_CONNECT, sock->sk, addr, sock->type);
}

int
gr_search_bind(const struct socket *sock, const struct sockaddr_in *addr)
{
	return gr_search_connectbind(GR_BIND, sock->sk, addr, sock->type);
}

int
gr_search_udp_sendmsg(const struct sock *sk, const struct sockaddr_in *addr)
{
	if (addr)
		return gr_search_connectbind(GR_CONNECT, sk, addr, SOCK_DGRAM);
	else {
		struct sockaddr_in sin;

		sin.sin_addr.s_addr = sk->daddr;
		sin.sin_port = sk->dport;

		return gr_search_connectbind(GR_CONNECT, sk, &sin, SOCK_DGRAM);
	}
}

int
gr_search_udp_recvmsg(const struct sock *sk, const struct sk_buff *skb)
{
	struct sockaddr_in sin;

	if (unlikely(skb->len < sizeof (struct udphdr)))
		return 1;	// skip this packet

	sin.sin_addr.s_addr = skb->nh.iph->saddr;
	sin.sin_port = skb->h.uh->source;

	return gr_search_connectbind(GR_CONNECT, sk, &sin, SOCK_DGRAM);
}
